###########################################################################################################
# Application Rshiny pour cartographier les risques d'impacts cumulés nationaux et régionaux
###########################################################################################################
# Script de chargement des packages, des données, des fonctions nécessaires à l'application
# Script de lancement de l'application 

# Données d'entrée : departements-20180101.shp ; Maillage_Metropole_Terrestre_10X10.shp ; regions-20180101.shp ; PointsChauds.xlsx ; PointsChaudsRegionaux.xlsx ; 
# Nombre_taxonsSCAP_sensibles_PressionsDT-Cumuls_Milieux_Spatial-Maille10X10.xlsx ; VARIABLES_PRESSIONS_COMPLET.csv

# Fichiers de sortie : fichiers png (image des cartes)

# Fonctions utilisées : mf_arrow ; plotmap.national ; plotmap.regional ; server
###########################################################################################################
###########################################################################################################

# LECTURE ET CHARGEMENT DES PACKAGES 
require(shiny)
require(dplyr)
require(readxl)
require(rgdal)
require(rgeos)
require(raster)
require(cartography)
require(mapsf)
require(sf)
require(magick)

library(shiny)
library(dplyr)
library(readxl)
library(rgdal)
library(rgeos)
library(raster)
library(cartography)
library(mapsf)
library(sf)
library(magick)

#  ui permet de générer l'affichage de l'application et les différents blocs qui la compose
ui <- fluidPage(
  
  # Titre de l'application
  titlePanel("Cartographie des risques d'impacts cumulés"),
  
  # Bar latérale
  sidebarPanel(
    
    # Entrée: Sélection de variable
    selectInput("exclu.ap", "Mailles contribuant fortement (CBG ≥ 80%) à la biodiversité générale en dehors des :", 
                c("Aires protégées fortes" = "AP_FORTE",
                  "Aires protégées fortes + N2000" = "AP_FORTEN2000",
                  "Toutes les aires protégées" = "AP_AICHI", 
                  "Pas d'exclusion de maille par rapport aux aires protégées" = "O")),
    
    selectInput("mlx", "Milieux :", 
                c("Tous les milieux" = "TousMlx",
                  "Forestiers" = "Forestiers",
                  "Agropastoraux" = "Agropastoraux", 
                  "Aquatiques" = "Aquatiques", 
                  "Humides" = 'Humides', 
                  "Rocheux" = "Rocheux", 
                  "Littoraux" = "Littoraux")),
    
    
    selectInput("press", "Pression :", 
                c("Agriculture intensive" = "AGRIC_INTENS",
                  "Sylviculture" = "SYLVIC",
                  "Urbanisation" = "URBA", 
                  "Espèces exotiques envahissantes" = "EEE", 
                  "Fréquentation et dérangement" = 'FREQ_DERANG')),
    
    selectInput("region", "Région :", 
                c("Toutes les régions métropolitaines" = "all",
                  "Auvergne Rhône-Alpes" = "Auvergne-Rhone-Alpes",
                  "Bourgogne-Franche-Comté" = "Bourgogne-Franche-Comte", 
                  "Bretagne" = "Bretagne", 
                  "Centre-Val de Loire" = 'Centre-Val-de-Loire', 
                  "Corse" = "Corse", 
                  "Grand Est" = "Grand-Est", 
                  "Hauts-de-France" = "Hauts-de-France", 
                  "Île-de-France" = "Ile-de-France", 
                  "Normandie" = "Normandie", 
                  "Nouvelle-Aquitaine" = "Nouvelle-Aquitaine", 
                  "Occitanie" = "Occitanie", 
                  "Pays de la Loire" = "Pays-de-la-Loire", 
                  "Provence-Alpes-Côte d'azur" = "Provence-Alpes-Cote-d-azur")),
    
    # Bouton de téléchargement
    downloadButton('ExportPlot', 'Exporter la carte (png)')
    
  ),
  
  # Panel principal
  mainPanel(
    imageOutput("rip_map"),
  )
  
)

# IMPORT ET MISE EN COMMUN DES TABLES 
CBG = read_excel(here::here('data','PointsChauds.xlsx'), sheet = 2)
CBG.region = read_excel(here::here('data','PointsChaudsRegionaux.xlsx'), sheet = 2, col_types = c('text', rep("numeric", 326)))
Nesp = read_excel(here::here('data','Nombre_taxonsSCAP_sensibles_PressionsDT-Cumuls_Milieux_Spatial-Maille10X10.xlsx'))
all.var = read.csv(here::here('data','VARIABLES_PRESSIONS_COMPLET.csv'), header = T)
myData = left_join(Nesp, all.var, by = 'CD_SIG')
myData.nat = left_join(myData, CBG, by = 'CD_SIG')
myData.reg = left_join(myData, CBG.region, by = 'CD_SIG')
press = read_excel(here::here('data','Liste_Croisements_Pressions.xlsx'), sheet = 2)
press = press[press$P_TYPE == 'CUMUL',]

####### LECTURE DES DONNEES SIG  #####
# MAILLES
maille = readOGR(here::here('data','Maillage_Metropole_Terrestre_10X10.shp'),'Maillage_Metropole_Terrestre_10X10')
maille.map.full = st_as_sf(maille)
# DEPARTEMENTS POUR SUPERPOSER
dep = readOGR(here::here('data','departements-20180101.shp'),'departements-20180101')
dep = spTransform(dep, maille@proj4string)
dep = raster::crop(dep, maille)
dep.map = st_as_sf(dep)
# REGION pour CROPPER 
region = st_read(here::here('data','regions-20180101.shp'), 'regions-20180101')
region$nom.lb = c(NA, 'Corse', NA, 'Ile-de-France', 'Hauts-de-France', "Nouvelle-Aquitaine", "Normandie" ,NA, "Pays-de-la-Loire", "Centre-Val-de-Loire",       
                  "Grand-Est" , NA, "Provence-Alpes-Cote-d-azur" , NA, "Bretagne", "Bourgogne-Franche-Comte", "Occitanie", "Auvergne-Rhone-Alpes")
region = st_transform(region, st_crs(dep.map))
maille.map.full = st_transform(maille.map.full, st_crs(dep.map))

# PREPARATION DE CHAQUE COUCHE REGIONALE DES DEPARTEMENTS ET DES MAILLES POUR ACCELERER L'APPLI 
for (r in unique(region$nom.lb)[!is.na(unique(region$nom.lb))]) {
  subregion = region[region$nom.lb  %in% r, ]
  
  # DEPARTEMENTS
  subdep = st_intersection(dep.map, subregion)
  types <- vapply(sf::st_geometry(subdep), function(x) {
    class(x)[2]
  }, "")
  subdep = subdep[types %in% c("MULTIPOLYGON" , "POLYGON"),] #retire les quelques lignes 
  assign(paste('subdep', r, sep= '.'), subdep)
  
  # MAILLES
  submaille = st_intersection(maille.map.full, subregion)
  assign(paste('submaille', r, sep= '.'), submaille)
}

# LEGENDE DES CARTES
bks = c("Pression faible - Nombre d'espèces vulnérables faible",
        "Pression faible - Nombre d'espèces vulnérables moyen",
        "Pression faible - Nombre d'espèces vulnérables élevé",
        "Pression moyenne - Nombre d'espèces vulnérables faible",
        "Pression moyenne - Nombre d'espèces vulnérables moyen",
        "Pression moyenne - Nombre d'espèces vulnérables élevé",
        "Pression forte - Nombre d'espèces vulnérables faible",
        "Pression forte - Nombre d'espèces vulnérables moyen",
        "Pression forte - Nombre d'espèces vulnérables élevé")
cols <- carto.pal(pal1 = "green.pal", # first color gradient
                  n1 = 3, transparency = T)
cols = c(cols, carto.pal(pal1 = "purple.pal",  n1 = 3, transparency = T))
cols = c(cols, carto.pal(pal1 = "red.pal",  n1 = 8, transparency = F)[c(2,5,8)])
cols.df = data.frame(bks = bks, cols = cols)

# LABELS DES CARTES
lb = paste('AP :', c('Aucune aire protégée', 'Protections fortes', 'Protections fortes + N2000', 'Toutes'))
lb.ap = data.frame(cd = c('O', 'AP_FORTE', 'AP_FORTEN2000', 'AP_AICHI'), lb = lb)

lb = paste('Milieux :', c('', 'Forestiers', 'Agropastoraux', 'Aquatiques', 'Humides', 'Rocheux', 'Littoraux'))
lb[1] = ''
lb.mlx = data.frame(cd = c('TousMlx', 'Forestiers', 'Agropastoraux', 'Aquatiques', 'Humides', 'Rocheux', 'Littoraux'), lb = lb)

lb.p = data.frame(cd = press$P_TABLE, lb = paste('Pression :', press$P_TITLE))
lb = paste('Région :', c("Auvergne Rhône-Alpes", "Bourgogne-Franche-Comté", "Bretagne", "Centre-Val de Loire", "Corse", "Grand Est", "Hauts-de-France",
                         "Île de France" , "Normandie" , "Nouvelle-Aquitaine" ,"Occitanie" , "Pays de la Loire", "Provence-Alpes-Côte d'azur"))
lb.r = data.frame(cd = c("Auvergne-Rhone-Alpes", "Bourgogne-Franche-Comte", "Bretagne", "Centre-Val-de-Loire", "Corse", "Grand-Est", "Hauts-de-France",
                         "Ile-de-France" , "Normandie" , "Nouvelle-Aquitaine" ,"Occitanie" , "Pays-de-la-Loire", "Provence-Alpes-Cote-d-azur"), 
                  lb = lb)

# THEME DES CARTES
custom <- list(
  name = "custom",
  bg = "#EDF4F5",
  fg = "#EDF4F5",
  mar = c(0, 0, 2, 0),
  tab = T,
  pos = "right",
  inner = TRUE,
  line = 10,
  cex = 1.5,
  font = 3
)

###########################
# LECTURE DES FONCTIONS
###########################
#' mf_arrow est une fonction disponible dans le package MAPSF qui ajoute une boussole sur une carte, ici elle a été modifiée pour pouvoir changer la taille de la pointe (ssize)
#'
#' @param pos : position de la pointe nord 
#' @param col : couleur de la pointe nord
#' @param adjust : ajustement de la direction de la pointe au vrai nord 
#' @param ssize : taille de la pointe 
#'
mf_arrow = function (pos = "topleft", col, adjust, ssize) 
{
  if (missing(col)) {
    col <- .gmapsf$args$fg
  }
  azim <- "N"
  theta <- 0
  mapExtent <- par("usr")
  xe <- mapExtent[1:2]
  ye <- mapExtent[3:4]
  inset <- strwidth("M", units = "user", cex = ssize)
  if (is.numeric(pos) & length(pos) == 2) {
    xarrow <- pos[1]
    yarrow <- pos[2]
  }
  else {
    switch(pos, topleft = {
      xarrow <- xe[1] + inset * 0.5
      yarrow <- ye[2] - inset * 1.5
    }, bottomright = {
      xarrow <- xe[2] - inset * 1.5
      yarrow <- ye[1] + inset * 1.5
    }, topright = {
      xarrow <- xe[2] - inset * 1.5
      yarrow <- ye[2] - inset * 1.5
    }, bottomleft = {
      xarrow <- xe[1] + inset * 0.5
      yarrow <- ye[1] + inset * 1.5
    }, top = {
      xarrow <- xe[1] + diff(xe)/2 - inset * 0.5
      yarrow <- ye[2] - inset * 1.5
    }, bottom = {
      xarrow <- xe[1] + diff(xe)/2 - inset * 0.5
      yarrow <- ye[1] + inset * 1.5
    }, left = {
      xarrow <- xe[1] + inset * 0.5
      yarrow <- ye[1] + diff(ye) * 0.5 - inset
    }, right = {
      xarrow <- xe[2] - inset * 1.5
      yarrow <- ye[1] + diff(ye) * 0.5 - inset
    })
  }
  xx <- c(xarrow, xarrow + inset/2, xarrow + inset)
  yy <- c(yarrow, yarrow + inset * 1, yarrow)
  if (!missing(adjust)) {
    xcrs <- st_crs(adjust)
    xp <- xarrow + inset/2
    yp <- yarrow + inset * 1.5
    A <- st_as_sf(x = data.frame(X = xp, Y = yp), coords = c("X", 
                                                             "Y"), crs = xcrs, remove = FALSE)
    B <- st_as_sf(x = data.frame(st_coordinates(A) + c(0, 
                                                       -1e+05)), coords = c("X", "Y"), crs = xcrs, 
                  remove = FALSE)
    Ap <- st_transform(A, 4326)
    Cp <- st_as_sf(x = data.frame(st_coordinates(Ap) + c(0, 
                                                         -2)), coords = c("X", "Y"), crs = 4326, 
                   remove = FALSE)
    C <- st_transform(Cp, xcrs)
    C[, c("X", "Y")] <- st_coordinates(C)
    DeltaXB <- B$X - A$X
    DeltaYB <- B$Y - A$Y
    DeltaXC <- C$X - A$X
    DeltaYC <- C$Y - A$Y
    VX <- c(DeltaXB, DeltaYB)
    VY <- c(DeltaXC, DeltaYC)
    theta <- acos(sum(VX * VY)/(sqrt(sum(VX * VX)) * sqrt(sum(VY * 
                                                                VY))))
    theta <- sign(C$X - B$X) * theta
    nc <- rot(A, c(xx[1], yy[1]), theta)
    xx[1] <- nc[1]
    yy[1] <- nc[2]
    nc <- rot(A, c(xx[3], yy[3]), theta)
    xx[3] <- nc[1]
    yy[3] <- nc[2]
  }
  polygon(xx, yy, col = col, border = col, xpd = TRUE)
  text(x = xx[1] + (xx[3] - xx[1])/2, y = yarrow, labels = azim, 
       adj = c(0.5, 1.3), cex = ssize*0.75, xpd = TRUE, font = 2, col = col, 
       srt = theta * 180/pi)
}

#' plotmap.national est une fonction qui produit la carte nationale des croisements pression / CBG pour un milieu selectionné. La fonction génère une image png qui est ensuite affichée dans l'application.
#'
#' @param ap : type de réseau d'aires protégées, sélectionné en entrée de l'application 
#' @param m : type de milieux, sélectionné en entrée de l'application 
#' @param p : type de pression, sélectionné en entrée de l'application 
#'
#' @return Une image png de la carte
plotmap.national = function(ap, m, p) {
  
  ###########################
  #### Préparation données 
  ###########################
  # Selection de la colonne CBG correspondante et des mailles avec CBG >= 80%
  cname = paste(ap, m, 'CBG', sep = '.') 
  cname = gsub('TousMlx.', '',cname)
  idx = myData.nat[, c("CD_SIG", cname)]
  idx = na.omit(idx)
  idx = idx$CD_SIG[idx[,2] >=80]
  
  # Sélection de la pression et du nombre d'espèces vulnérables
  cd.ly = press$CD_LY[press$P_TABLE == p]
  val_press = myData.nat[, grep(paste0('VAR_', cd.ly), colnames(myData.nat))]
  Nsensi = myData.nat[, grep(cd.ly, colnames(myData.nat))]
  Nsensi = Nsensi[, grep('Ntaxon_sensi', colnames(Nsensi))]
  Nsensi = Nsensi[, grep(m, colnames(Nsensi))]
  Nsensi = Nsensi[, grep(p, colnames(Nsensi))]
  
  # Catégorisation pression
  th.1 = quantile(unlist(na.omit(val_press)), probs = 0.33)
  th.2 = quantile(unlist(na.omit(val_press)), probs = 0.66)
  val_press_cat = ifelse(val_press <=th.1, 'Pression faible', 
                         ifelse(val_press <=th.2, 'Pression moyenne', 'Pression forte'))
  
  # Catégorisation N. vulnérables
  th.1 = quantile(unlist(Nsensi), probs = 0.33)
  th.2 = quantile(unlist(Nsensi), probs = 0.66)
  val_Nsensi_cat = ifelse(Nsensi <=th.1, 'Nombre d\'espèces vulnérables faible', 
                          ifelse(Nsensi <=th.2, 'Nombre d\'espèces vulnérables moyen', 'Nombre d\'espèces vulnérables élevé'))
  
  # Mise en commun
  subdata = data.frame(CD_SIG = myData.nat$CD_SIG, Nsensi,  val_Nsensi_cat, 
                       val_press, val_press_cat)
  colnames(subdata) = c('CD_SIG', 'Nsensi', 'Nsensi_cat', 'val_pression', 'val_pression_cat')
  subdata = subdata[subdata$CD_SIG %in% idx,]
  
  # Jointure aux mailles et creation de la carte 
  maille.map = maille.map.full
  maille.map = left_join(maille.map, subdata, by = 'CD_SIG')
  maille.map$cat.map = paste(maille.map$val_pression_cat, maille.map$Nsensi_cat, sep = ' - ') 
  
  ###########################
  #### Cartographie 
  ###########################
  
  # Ajustement de la légende
  cols.df.i = cols.df[cols.df$bks %in% unique(maille.map$cat.map),] 
  toadd = unique(maille.map$cat.map)
  toadd = toadd[grep("NA", toadd)]
  cols.df.i2 = rbind(cols.df.i, data.frame(bks = 'Hors points chauds', cols = 'grey'))
  cols.df.i = rbind(cols.df.i, data.frame(bks = toadd, cols = 'grey'))
  
  # Création fichier de sortie
  outfile <- tempfile(fileext='.png')
  # Création titre de la carte 
  lb.title = ifelse(m == 'TousMlx', paste(lb.p$lb[lb.p$cd == p],lb.ap$lb[lb.ap$cd == ap], sep = '\n'), paste(lb.p$lb[lb.p$cd == p],lb.ap$lb[lb.ap$cd == ap], lb.mlx$lb[lb.mlx$cd == m], sep = '\n'))
  
  png(outfile, width=1500, height=900)
  map.i = mf_init(x = maille.map, theme = custom, expandBB = c(0,0,0,.6))
  map.i = mf_map(x = maille.map,  var = "cat.map", type = "typo" , pal = cols.df.i$cols, val_order = cols.df.i$bks,
                 add = TRUE, border = NA, leg_pos = NA)
  map.i = mf_base(x = dep.map, add = T, col = NA, lwd = 2, border = 'black')
  map.i = mf_legend_t(pos = 'right', val = cols.df.i2$bks, pal = cols.df.i2$cols,  title = 'Catégories', cex = 1.4, 
                      val_cex = 1.4, title_cex = 1.4, fg = '#82888A')
  # Plot credits
  map.i = mf_credits(paste0('Patrinat, ',substr(Sys.time(),1,4)), cex = 1.3, col = '#82888A')
  # Plot a scale bar
  map.i = mf_scale(size = 200, cex = 1.3, col = '#82888A')
  # Plot a north arrow
  map.i = mf_arrow('topleft', col = 'black', ssize = 2)
  # Plot a title
  map.i = mf_title(lb.title, fg = '#82888A')
  
  dev.off()
  
  # Liste pour le fonctionnement de l'appli
  return(list(src = outfile,
              alt = "En cours de création"))
  
}

#' plotmap.regional est une fonction qui produit la carte régionale des croisements pression / CBG pour un milieu selectionné. La fonction génère une image png qui est ensuite affichée dans l'application.
#'
#' @param ap : type de réseau d'aires protégées, sélectionné en entrée de l'application
#' @param m : type de milieux, sélectionné en entrée de l'application
#' @param p : type de pression, sélectionné en entrée de l'application
#' @param r : région affichée, sélectionnée en entrée de l'application
#'
#' @return Une image png de la carte
plotmap.regional = function(ap, m, p, r) {
  
  ###########################
  #### Préparation données 
  ###########################
  # Selection région et département 
  subregion = region[region$nom.lb  %in% r, ]
  subdep = get(paste('subdep', r, sep= '.'))
  
  # Selection de la colonne CBG correspondante et des mailles avec CBG >= 80%
  cname = paste(ap, m, r, 'CBG', sep = '.') 
  cname = gsub('TousMlx.', '',cname)
  idx = try(myData.reg[, c("CD_SIG", cname)])
  
  if (sum(class(idx) %in% 'try-error')!=1) {
    
    idx = na.omit(idx)
    idx = idx$CD_SIG[idx[,2] >=80]
    
    # Sélection de la pression et du nombre d'espèces vulnérables
    cd.ly = press$CD_LY[press$P_TABLE == p]
    val_press = myData.reg[, grep(paste0('VAR_', cd.ly), colnames(myData.reg))]
    Nsensi = myData.reg[, grep(cd.ly, colnames(myData.reg))]
    Nsensi = Nsensi[, grep('Ntaxon_sensi', colnames(Nsensi))]
    Nsensi = Nsensi[, grep(m, colnames(Nsensi))]
    Nsensi = Nsensi[, grep(p, colnames(Nsensi))]
    
    # catégorisation pression
    th.1 = quantile(unlist(na.omit(val_press)), probs = 0.33)
    th.2 = quantile(unlist(na.omit(val_press)), probs = 0.66)
    val_press_cat = ifelse(val_press <=th.1, 'Pression faible', 
                           ifelse(val_press <=th.2, 'Pression moyenne', 'Pression forte'))
    
    # Catégorisation N. vulnérables
    th.1 = quantile(unlist(Nsensi), probs = 0.33)
    th.2 = quantile(unlist(Nsensi), probs = 0.66)
    val_Nsensi_cat = ifelse(Nsensi <=th.1, 'Nombre d\'espèces vulnérables faible', 
                            ifelse(Nsensi <=th.2, 'Nombre d\'espèces vulnérables moyen', 'Nombre d\'espèces vulnérables élevé'))
    
    # Mise en commun
    subdata = data.frame(CD_SIG = myData.reg$CD_SIG, Nsensi,  val_Nsensi_cat, 
                         val_press, val_press_cat)
    colnames(subdata) = c('CD_SIG', 'Nsensi', 'Nsensi_cat', 'val_pression', 'val_pression_cat')
    subdata = subdata[subdata$CD_SIG %in% idx,]
    
    # Jointure aux mailles et creation de la carte 
    maille.map = get(paste('submaille', r, sep= '.'))
    maille.map = left_join(maille.map, subdata, by = 'CD_SIG')
    maille.map$cat.map = paste(maille.map$val_pression_cat, maille.map$Nsensi_cat, sep = ' - ') 
    
    ###########################
    #### Cartographie 
    ###########################
    
    # Ajustement de la légende
    cols.df.i = cols.df[cols.df$bks %in% unique(maille.map$cat.map),] #garde que les couleurs necessaires
    toadd = unique(maille.map$cat.map)
    toadd = toadd[grep("NA", toadd)]
    cols.df.i2 = rbind(cols.df.i, data.frame(bks = 'Hors points chauds', cols = 'grey'))
    cols.df.i = rbind(cols.df.i, data.frame(bks = toadd, cols = 'grey'))
    
    # Création fichier de sortie
    outfile <- tempfile(fileext='.png')
    # Création titre de la carte 
    lb.title = ifelse(m == 'TousMlx', paste(lb.p$lb[lb.p$cd == p],lb.ap$lb[lb.ap$cd == ap], lb.r$lb[lb.r$cd == r], sep = '\n'), paste(lb.p$lb[lb.p$cd == p],lb.ap$lb[lb.ap$cd == ap], lb.mlx$lb[lb.mlx$cd == m], lb.r$lb[lb.r$cd == r], sep = '\n'))
    
    png(outfile, width=1500, height=900)
    map.i = mf_init(x = maille.map, theme = custom, expandBB = c(0,0,0,.6))
    map.i = mf_map(x = maille.map,  var = "cat.map", type = "typo" , pal = cols.df.i$cols, val_order = cols.df.i$bks,
                   add = TRUE, border = NA, leg_pos = NA)
    map.i = mf_base(x = subdep, add = T, col = NA, lwd = 2, border = 'black')
    map.i = mf_legend_t(pos = 'right', val = cols.df.i2$bks, pal = cols.df.i2$cols,  title = 'Catégories', cex = 1.4, 
                        val_cex = 1.4, title_cex = 1.4, fg = '#82888A')
    # Plot credits
    map.i = mf_credits(paste0('Patrinat, ',substr(Sys.time(),1,4)), cex = 1.3, col = '#82888A')
    # Plot a scale bar
    map.i = mf_scale(size = 50, cex = 1.3, col = '#82888A')
    # Plot a north arrow
    map.i = mf_arrow('topleft', col = 'black', ssize = 2)
    # Plot a title
    map.i = mf_title(lb.title, fg = '#82888A')
    
    dev.off()
  } else {
    outfile <- tempfile(fileext='.png')
    png(outfile, width=1500, height=900)
    dev.off()
    img = image_read(outfile)
    img = image_annotate(img, "Cartographie non réalisable en vue des paramètres", size = 40, gravity = "center", color = "Black")
    image_write(img, path = outfile, format = "png")
  }
  # Liste pour le fonctionnement de l'appli
  return(list(src = outfile,
              alt = "En cours de création"))
}

#' server est une fonction qui génère ce qu'il y aura dans l'application
#'
#' @param input : les paramètres d'entrée de l'application (ceux choisis par l'utilisateur)
#' @param output : les paramètres de sortie de l'application (ce que l'application affiche)
#'
server <- function(input, output) {
  
  
  output$rip_map <- renderImage({
    if (input$region == 'all') {
      plotmap.national(ap = input$exclu.ap, m = input$mlx, p = input$press)
    } else {
      plotmap.regional(ap = input$exclu.ap, m = input$mlx, p = input$press, r = input$region)
    }
  }, deleteFile = T)
  
  
  # download
  output$ExportPlot <- downloadHandler(
    
    # file name
    filename <- function(ap = input$exclu.ap, m = input$mlx, p = input$press, r = input$region) {
      if (input$region == 'all') {
        paste0('Cartographie-RIP-Nationale_Exclusion-',ap,'_Milieux-', m, '_Pression-',p, '.png')
      } else {
        paste0('Cartographie-RIP-Regionale_Exclusion-',ap,'_Milieux-', m, '_Pression-',p, '_Region-',r,'.png')
      }
    },
    
    # content
    content = function(file){
      if (input$region == 'all') {
        file.copy(plotmap.national(ap = input$exclu.ap, m = input$mlx, p = input$press)$src, file)
      } else {
        file.copy(plotmap.regional(ap = input$exclu.ap, m = input$mlx, p = input$press, r = input$region)$src, file)
      }
    },    
    
    # Type de sortie
    contentType = "image/png"
  )
}

# Run the application 
shinyApp(ui = ui, server = server)
